# Node.js on Fly.io

## Recommended Dockerfile

```dockerfile
FROM node:20-alpine

WORKDIR /app

# Install dependencies first (cache layer)
COPY package*.json ./
RUN npm ci --only=production

# Copy application
COPY . .

# Create non-root user
RUN addgroup --system --gid 1001 nodejs && \
    adduser --system --uid 1001 nodeuser
USER nodeuser

ENV NODE_ENV=production
ENV PORT=8080
EXPOSE 8080

CMD ["node", "server.js"]
```

## fly.toml

```toml
app = "my-node-app"
primary_region = "ord"

[build]
  dockerfile = "Dockerfile"

[env]
  NODE_ENV = "production"
  PORT = "8080"

[http_service]
  internal_port = 8080
  force_https = true
  auto_stop_machines = "stop"
  auto_start_machines = true

[[http_service.checks]]
  grace_period = "10s"
  interval = "30s"
  method = "GET"
  path = "/health"
  timeout = "5s"
```

## Express.js Example

```js
const express = require('express');
const app = express();

const PORT = process.env.PORT || 8080;
const HOST = '0.0.0.0';  // Important: bind to all interfaces

app.get('/health', (req, res) => {
  res.status(200).send('OK');
});

app.get('/', (req, res) => {
  res.send('Hello from Fly.io!');
});

app.listen(PORT, HOST, () => {
  console.log(`Server running on ${HOST}:${PORT}`);
});
```

## Fastify Example

```js
const fastify = require('fastify')({ logger: true });

fastify.get('/health', async () => ({ status: 'ok' }));
fastify.get('/', async () => ({ hello: 'world' }));

const start = async () => {
  await fastify.listen({
    port: process.env.PORT || 8080,
    host: '0.0.0.0'  // Important
  });
};

start();
```

## TypeScript Projects

Dockerfile for TypeScript:

```dockerfile
FROM node:20-alpine AS builder
WORKDIR /app
COPY package*.json ./
RUN npm ci
COPY . .
RUN npm run build

FROM node:20-alpine
WORKDIR /app
ENV NODE_ENV=production
COPY --from=builder /app/dist ./dist
COPY --from=builder /app/node_modules ./node_modules
COPY package*.json ./

USER node
EXPOSE 8080
CMD ["node", "dist/index.js"]
```

## .dockerignore

```
node_modules
.git
*.md
.env*
coverage
dist
.nyc_output
```

## Common Issues

### App Binds to localhost

**Symptom:** Works locally, fails health checks on Fly.

**Fix:** Bind to `0.0.0.0`:
```js
app.listen(PORT, '0.0.0.0');
```

### ES Modules

If using ES modules (`"type": "module"` in package.json):
```dockerfile
CMD ["node", "--experimental-specifier-resolution=node", "server.js"]
```

### Native Dependencies

Some packages need build tools:
```dockerfile
FROM node:20-alpine
RUN apk add --no-cache python3 make g++
# ... rest of Dockerfile
```

### Graceful Shutdown

Handle SIGTERM for clean shutdown:
```js
process.on('SIGTERM', () => {
  console.log('SIGTERM received, shutting down...');
  server.close(() => {
    process.exit(0);
  });
});
```
